#include <tbb/task_group.h>
#include <tbb/global_control.h>
#include <iostream>
#include <cstdlib>
#include <thread>   // for hardware_concurrency()

// Naive parallel Fibonacci using task_group
long fib(int n) {
    if (n < 2) return n;

    long x = 0, y = 0;
    tbb::task_group g;

    g.run([&]{ x = fib(n - 1); }); // spawn task
    g.run([&]{ y = fib(n - 2); }); // spawn task
    g.wait();                      // wait both

    return x + y;
}

int main(int argc, char** argv) {
    if (argc < 2) {
        std::cerr << "usage: " << argv[0] << " N [TBB_NUM_THREADS via env]\n";
        return 1;
    }
    int n = std::atoi(argv[1]);

    // Choose worker count: env overrides hardware_concurrency
    std::size_t w = std::thread::hardware_concurrency();
    if (const char* tnt = std::getenv("TBB_NUM_THREADS")) {
        int tmp = std::atoi(tnt);
        if (tmp > 0) w = static_cast<std::size_t>(tmp);
    }
    if (w == 0) w = 1;

    // Cap max parallelism
    tbb::global_control gc(tbb::global_control::max_allowed_parallelism, w);
    std::cout << "w=" << w << "\n";

    long sum = fib(n);
    std::cout << "Fibonacci of " << n << " is " << sum << "\n";
    return 0;
}

