#pragma once
#include <stdio.h>
#include <stdlib.h>
#include <cuda_runtime.h>

/* Opaque handle type like the old SDK: user holds StopWatchInterface* */
typedef struct {
    cudaEvent_t start_evt;
    cudaEvent_t stop_evt;
    int         running;
    float       elapsed_ms;
} StopWatchInterface;

/* ================= Timer API (matches old signatures) ===================== */
/* Create: user passes &hTimer, where hTimer is StopWatchInterface* */
static inline int cutCreateTimer(StopWatchInterface **t)
{
    if (!t) return 1;
    StopWatchInterface *p = (StopWatchInterface*)calloc(1, sizeof(*p));
    if (!p) return 1;
    cudaEventCreate(&p->start_evt);
    cudaEventCreate(&p->stop_evt);
    p->running    = 0;
    p->elapsed_ms = 0.0f;
    *t = p;
    return 0; /* success */
}

/* Delete: user passes &hTimer */
static inline void cutDeleteTimer(StopWatchInterface **t)
{
    if (!t || !*t) return;
    if ((*t)->start_evt) cudaEventDestroy((*t)->start_evt);
    if ((*t)->stop_evt)  cudaEventDestroy((*t)->stop_evt);
    free(*t);
    *t = NULL;
}

/* Reset */
static inline void cutResetTimer(StopWatchInterface *t)
{
    if (!t) return;
    if (t->start_evt) cudaEventDestroy(t->start_evt);
    if (t->stop_evt)  cudaEventDestroy(t->stop_evt);
    cudaEventCreate(&t->start_evt);
    cudaEventCreate(&t->stop_evt);
    t->running    = 0;
    t->elapsed_ms = 0.0f;
}

/* Start/Stop/Get */
static inline void cutStartTimer(StopWatchInterface *t)
{
    if (!t) return;
    cudaEventRecord(t->start_evt, 0);
    t->running = 1;
}

static inline void cutStopTimer(StopWatchInterface *t)
{
    if (!t) return;
    cudaEventRecord(t->stop_evt, 0);
    cudaEventSynchronize(t->stop_evt);
    float ms = 0.0f;
    cudaEventElapsedTime(&ms, t->start_evt, t->stop_evt);
    t->elapsed_ms = ms;
    t->running = 0;
}

static inline float cutGetTimerValue(StopWatchInterface *t)
{
    return t ? t->elapsed_ms : 0.0f;
}

/* ================= Minimal error helpers to satisfy old macros ============ */
#ifndef cutilSafeCall
#  define cutilSafeCall(expr) do {                                    \
        cudaError_t __e = (expr);                                     \
        if (__e != cudaSuccess) {                                     \
            fprintf(stderr, "[CUDA] %s failed: %s\n", #expr,           \
                    cudaGetErrorString(__e));                         \
            exit(EXIT_FAILURE);                                       \
        }                                                             \
    } while(0)
#endif

#ifndef cutilCheckError
#  define cutilCheckError(expr) do {                                   \
        (void)(expr);                                                  \
        cudaError_t __e = cudaGetLastError();                          \
        if (__e != cudaSuccess) {                                      \
            fprintf(stderr, "[CUDA] runtime error: %s\n",               \
                    cudaGetErrorString(__e));                          \
            exit(EXIT_FAILURE);                                        \
        }                                                              \
    } while(0)
#endif

#ifndef cutilDeviceSynchronize
#  define cutilDeviceSynchronize cudaDeviceSynchronize
#endif

