`timescale 1ns/1ps
module cpu_testbench;
   reg reset_; initial begin reset_=0; #22 reset_=1; #400; $stop; end
   reg clock;  initial clock=0; always #5 clock<=(!clock);
   wire[31:0] PC,IR,D,Q; wire RESET; assign PC=cpu.mycore.dp.pc, IR=cpu.mycore.dp.instr;
   assign RESET=cpu.mycore.dp.pcreg.reset_,Q=cpu.mycore.dp.pcreg.q,D=cpu.mycore.dp.pcreg.d;
   wire[31:0] A,B,RESULT; assign A=cpu.mycore.dp.srca,B=cpu.mycore.dp.srcb,RESULT=cpu.mycore.dp.result;
   initial begin wait(reset_); #180
      $display("R2=%h  (should contain '00000007')",cpu.mycore.dp.rf.rf[2]);
      $finish;
   end
   mips cpu(clock,reset_);
endmodule
   
module mips(clk,reset_);
   input clk,reset_; wire memwrite;
   wire [31:0] readdata,instr,pc,aluout,writedata;
   // instantiate processor and memories
   core mycore(clk,reset_,readdata,instr,
               pc,memwrite,aluout,writedata);
   imem imem(pc[7:2],  instr);
   dmem dmem(clk,memwrite,aluout[7:0],writedata,  readdata);
endmodule

module core(clk,reset_,readdata,instr,
            pc, memwrite, aluout, writedata);
   input clk, reset_; input[31:0] readdata,instr;
   output[31:0] pc,aluout,writedata;
   output memwrite;
   wire memtoreg,alusrc,regdst,regwrite,pcsrc,
   zero,overflow; wire[2:0] alucontrol;

   controller c(instr[31:26],instr[5:0],zero,
                  memtoreg,memwrite,pcsrc,alusrc,
                  regdst,regwrite,alucontrol);
   datapath dp(clk,reset_,memtoreg,pcsrc,alusrc,
         regdst,regwrite,alucontrol,instr,readdata,
         zero,overflow,pc,aluout,writedata);
endmodule

module controller(opcode,funct,zero,  
          memtoreg,memwrite,pcsrc,alusrc,
          regdst,regwrite,alucontrol);
   input[5:0] opcode, funct; input zero;
   output memtoreg,memwrite,pcsrc,
          alusrc,regdst,regwrite;
   output [2:0] alucontrol;
   wire [1:0] aluop; wire branch;

   maindec md(opcode,  memtoreg,memwrite,
              branch,alusrc,regdst,regwrite,aluop);
   aludec ad(funct,aluop,  alucontrol);
   assign pcsrc = branch & zero;
endmodule

module maindec(opcode,  memtoreg,memwrite,branch,
               alusrc,regdst,regwrite,aluop);
   input [5:0] opcode;
   output memtoreg,memwrite,branch,alusrc,regdst, regwrite;
   output [1:0] aluop;
   reg [7:0] controls;
   assign {regwrite,regdst,alusrc,branch,
           memwrite,memtoreg,aluop} = controls;
   always @(opcode)
   casex(opcode)
      6'b000000: controls <= 8'b11000010; // RTYPE
      6'b100011: controls <= 8'b10100100; // LW
      6'b101011: controls <= 8'b00101000; // SW
      6'b000100: controls <= 8'b00010001; // BEQ
      6'b001000: controls <= 8'b10100000; // ADDI
      default:   controls <= 8'bxxxxxxxx; // illegal op
   endcase
endmodule

module aludec(funct,aluop,  alucontrol);
   input [5:0] funct; input [1:0] aluop; 
   output [2:0] alucontrol;
   reg [2:0] alucontrol;
   always @(aluop or funct)
   casex(aluop)
      2'b00: alucontrol <= 3'b010; // add (for lw/sw/addi)
      2'b01: alucontrol <= 3'b110; // sub (for beq)
      default: casex(funct) // R-type instructions
         6'b100000: alucontrol <= 3'b010; // add
         6'b100010: alucontrol <= 3'b110; // sub
         6'b100100: alucontrol <= 3'b000; // and
         6'b100101: alucontrol <= 3'b001; // or
         6'b101010: alucontrol <= 3'b111; // slt
         default: alucontrol <= 3'bxxx; // ???
      endcase
   endcase
endmodule

module datapath(clk, reset_, memtoreg, pcsrc, alusrc, regdst, regwrite,
                alucontrol, instr, readdata,
                zero, overflow, pc, aluout, writedata);
   input clk, reset_, memtoreg, pcsrc, alusrc, regdst, regwrite;
   input[2:0] alucontrol; input[31:0] instr, readdata;
   output zero, overflow; output[31:0] pc, aluout, writedata;
   wire[4:0] writereg; wire[31:0] pcnext, pcplus4, pcbranch;
   wire[31:0] signimm, signimmsh, srca, srcb, result;
   // next PC logic
   flopr #(32) pcreg(clk, reset_, pcnext, pc);
   adder       pcadd1(pc, 32'b100, pcplus4);
   sl2         immsh(signimm, signimmsh);
   adder       pcadd2(pcplus4, signimmsh, pcbranch);
   mux2 #(32)  pcbrmux(pcplus4, pcbranch, pcsrc, pcnext);
   // register file logic
   regfile     rf(clk,regwrite, instr[25:21], instr[20:16], writereg,
                  result, srca, writedata);
   mux2 #(5)   wrmux(instr[20:16], instr[15:11], regdst, writereg);
   mux2 #(32)  resmux(aluout, readdata, memtoreg, result);
   signext     se(instr[15:0], signimm);
   // ALU logic
   mux2 #(32)  srcbmux(writedata, signimm, alusrc, srcb);
   alu         alu(srca, srcb, alucontrol, aluout, zero, overflow);
endmodule

module regfile(clk, we3, ra1, ra2, wa3, wd3,   rd1, rd2);
   input clk,we3; input[4:0] ra1, ra2, wa3; input[31:0] wd3;
   output[31:0] rd1, rd2; reg[31:0] rd1, rd2;
   reg[31:0] rf[0:31];
   // three ported register file: read two ports combinationally
   // write third port on falling edge of clk; register 0 hardwired to 0
   always @(negedge clk) if (we3) rf[wa3] <= wd3;
   always @(ra1) rd1 <= (ra1 != 0) ? rf[ra1] : 0; 
   always @(ra2) rd2 <= (ra2 != 0) ? rf[ra2] : 0;
endmodule
module adder(a, b,  aluout);
   input[31:0] a, b;
   output[31:0] aluout;
   assign aluout=a+b;
endmodule

module alu(a, b, aluctrl,  aluout, zero, overflow);
   input[31:0] a, b; input[2:0] aluctrl;
   output[31:0] aluout; output zero, overflow; reg[31:0] aluout;
   assign zero = (aluout==0); //v------ delay to av. inf. speed loops
   always @(aluctrl or a or b) #0.1 //re-evaluate if these change
      casex (aluctrl)
         0: aluout <= a & b;
         1: aluout <= a | b;
         2: aluout <= a + b;
         6: aluout <= a - b;
         7: aluout <= a<b ? 1:0;
         default: aluout<=0; //default to 0, should not happen
      endcase
endmodule

module adder(a, b,  aluout);
   input[31:0] a, b;
   output[31:0] aluout;
   assign aluout=a+b;
endmodule

module sl2(a, y); // shift left by 2
  input [31:0] a; output [31:0] y;
   assign y = {a[29:0], 2'b00};
endmodule

module signext(a, y);
   input [15:0] a; output [31:0] y;
   assign y = {{16{a[15]}}, a};
endmodule

module flopr (clk, reset_, d, q);
   parameter WIDTH = 8;
   input  [WIDTH-1:0] d; input clk, reset_;
   output [WIDTH-1:0] q; reg [WIDTH-1:0] q;
   always @(posedge clk) // posedge reset)
      #1.1 if (!reset_) q <= 0; //delay inserted to avoid infinite speed loops
      else q <= d;
endmodule

module mux2 (d0, d1, s, y);
  parameter WIDTH = 32;
  input [WIDTH-1:0] d0, d1;
  input s;
  output [WIDTH-1:0] y;
  assign y = s ? d1 : d0;
endmodule

module imem(a, rd);
   input [5:0] a; output [31:0] rd;
   reg [31:0] RAM[0:63];
   initial $readmemh("memfile.dat", RAM);
   assign rd = RAM[a]; // word aligned
endmodule

module dmem(clk, we, a, wd,   rd);
   input clk, we;  input[7:0] a; input[31:0] wd; output[31:0] rd;
   reg [31:0] RAM2[0:63];
   assign rd = RAM2[a[7:2]]; // word aligned
   always @(posedge clk)
   if (we) RAM2[a[7:2]] <= wd;
endmodule



